#!/usr/bin/env bash

set -e

if [ -z "$SVN_USERNAME" ]; then
  echo "Missing SVN username. Please pass it via env variable SVN_USERNAME"
  exit 1
fi

if [ -z "$SVN_PASSWORD" ]; then
  echo "Missing SVN password. Please pass it via env variable SVN_PASSWORD"
  exit 1
fi

DOCKER_NAME=wp-build
DOCKER_WORKING_DIR=/app
SVN_FOLDER=__svn__
SVN_URL="https://plugins.svn.wordpress.org/trinity-audio"

svn_exec() {
  if [ -z "$2" ]; then
    path=${SVN_FOLDER}
  else
    path=$2
  fi

  docker exec ${DOCKER_NAME} su node -c "cd $path && svn --non-interactive --no-auth-cache --username '${SVN_USERNAME}' --password '${SVN_PASSWORD}' $1"
}

version=$(grep 'Version:' trinity.php | cut -d ':' -f2 | xargs)

if [ -z "$version" ]; then
  echo "Can't find plugin version!"
  exit 1
else
  echo "Found plugin version: ${version}"
fi

echo "Build SCSS files..."

docker rm -f ${DOCKER_NAME}
docker run -td \
  --name ${DOCKER_NAME} \
  -v "$(pwd)":"${DOCKER_WORKING_DIR}" \
  -w ${DOCKER_WORKING_DIR} \
  node:16

docker exec ${DOCKER_NAME} chown -R node:node /usr/local

docker exec ${DOCKER_NAME} su node -c "npm i"
docker exec ${DOCKER_NAME} su node -c "npm run build"

echo "Checkout SVN code..."
rm -rf ${SVN_FOLDER}
svn_exec "checkout ${SVN_URL}/trunk ${SVN_FOLDER}" "${DOCKER_WORKING_DIR}"

echo "Copying files..."
cp ./*.php ./*.txt ${SVN_FOLDER}
cp -r migrations js inc admin dist ${SVN_FOLDER}
cp -r assets/fonts assets/icons assets/images ${SVN_FOLDER}/assets

diff=$(svn_exec "diff .")

if [ -z "$diff" ]; then
  echo "No diff was found!"
  exit 1
else
  echo "The following diff will be committed:"
  svn_exec "diff ."
fi

echo "Commit changes to trunk..."
svn_exec "commit -m 'v${version}'"

SVN_TAG_URL="${SVN_URL}/tags/v${version}"

if svn_exec "info $SVN_TAG_URL"; then
  echo "Tag already exists for version ${version}. Skipping it"
else
  echo "Making tag for version ${version}..."
  svn_exec "copy '${SVN_URL}/trunk' '${SVN_URL}/tags/v${version}' -m 'v${version}'"
fi

docker rm -f ${DOCKER_NAME}
rm -rf ${SVN_FOLDER}

echo "Done!"
